#### AllSeasons: Fig.A1 describe visit frequency by season  ###############
# Script author: Peter King (p.king1@leeds.ac.uk)
# Last Edited: 16/10/2025


# *****************************************************************************
#### Section 0: Setting up ####
## Find session information in 00_ReplicatePaper.R
# *****************************************************************************


## Libraries here: ************************************************************
library(tidyverse)
library(dplyr)
library(magrittr)
library(mded)
library(here)
library(data.table)
library(Rfast)
library(janitor)


# ***************************************************************************
#### Section 1: Setup parallelisation and benchmarking ####
# ***************************************************************************


## Use step three and add unconditionals rather than use Step4.csv with conditionals
AllSeasons <-
  here("SurveyData",
       "AllSeasons_dataframe_Step3_Anonymised.csv") %>%
  fread() %>%
  data.frame()


# ***************************************************************************
#### Section 2: Define useful variables ####
# ***************************************************************************


## So I'll use these later to name rows
Names <- c(
  "Tax",
  "ColourHigh",
  "ColourMedium",
  "SmellHigh",
  "SmellMedium",
  "SoundHigh",
  "SoundMedium",
  "DeadwoodHigh",
  "DeadwoodMedium"
)


Facet_Labels <- c(
  '0' = "Winter (N = 1879)",
  '1' = "Spring (N = 1765)",
  '2' = "Summer (N = 1767)",
  '3' = "Autumn (N = 1748)"
)



# ## New version
Labels <- c(
  "Tax",
  "Colours:\n high",
  "Colours:\n medium",
  "Smells:\n high",
  "Smells:\n medium",
  "Sounds:\n high",
  "Sounds:\n medium",
  "Deadwood:\ndecomposition\n high",
  "Deadwood:\ndecomposition\n medium"
)


## Label grouping variable
VisitFrequency <- c(0, 1, 2, 3, 4, 5)


LegendLabels <- c(
  paste0("I do not visit\n(N = ",AllSeasons %>% filter(MostRecentVisit==0) %>% nrow(),")"),
  paste0("Once or twice a season\n(N = " ,AllSeasons %>% filter(MostRecentVisit==1) %>% nrow(),")"),
  paste0("Once or twice a month\n(N = ",AllSeasons %>% filter(MostRecentVisit==2) %>% nrow(),")"),
  paste0("Once a week\n(N = " ,AllSeasons %>% filter(MostRecentVisit==3) %>% nrow(),")"),
  paste0("Several times a week\n(N = " ,AllSeasons %>% filter(MostRecentVisit==4) %>% nrow(),")"),
  paste0("Every day\n(N = " ,AllSeasons %>% filter(MostRecentVisit==5) %>% nrow(),")"))



LegendLabels_Simple <- c(
  '0' = "I do not visit",
  '1' = "Once or twice a season",
  '2' = "Once or twice a month",
  '3' = "Once a week",
  '4' = "Several times a week",
  '5' = "Every day"
)



## I got these from RColorBrewer but adding specifically here to
### (a) see the colours in the console with new RStudio,
### (b) make scale_fill_manual() easier
Colours <- c(
  "#C6DBEF",
  "#C6DBEF",
  "#08306B",
  "#08306B",
  "#6BAED6",
  "#6BAED6",
  "#2171B5",
  "#2171B5",
  "#F7FBFF"
)

## Update all text sizes here for consistency
TextSize <- 12

TextSetup <- element_text(size = TextSize,
                          colour = "black",
                          family = "serif")


## Reporting percentage values here
AllSeasons %>%
  janitor::tabyl(MostRecentVisit, Season) %>%
  janitor::adorn_percentages(denominator = "col") %>%
  janitor::adorn_pct_formatting(digits = 2)
# MostRecentVisit
# 0      1      2      3
# 0 39.12% 32.39% 28.47% 27.92%
# 1 13.62% 12.78% 14.60% 18.25%
# 2 15.17% 17.75% 17.66% 19.34%
# 3 14.42% 18.03% 17.20% 15.50%
# 4 12.56% 13.40% 16.64% 12.59%
# 5  5.11%  5.65%  5.43%  6.41%





# ***************************************************************************
#### Section 3: Create Plot ####
# ***************************************************************************


FigureA1 <-
AllSeasons[, c("Season",
               "MostRecentVisit")] %>%

  pivot_longer(
    cols = c(MostRecentVisit),
    names_to = "name",
    values_to = "value"
  ) %>%

  group_by(Season, value) %>%

  summarise(Freq = n(),
            .groups = "drop") %>%

  # Calculate the proportion of each level of MostRecentVisit for each Season
  group_by(Season) %>%

  mutate(FreqProp = Freq / sum(Freq)) %>%

  ungroup() %>%

  ggplot(aes(
    y = FreqProp,
    x = value,
    fill = Season %>% as.factor()
  )) +

  geom_bar(stat = "identity",
           position = "dodge",
           colour = "black",
           linewidth = 0.25) +

  theme_bw() +

  facet_wrap(~ value,
             scales = "free_x",
             labeller = as_labeller(LegendLabels_Simple)) +

  scale_fill_manual(
    name = "Season",
    values = RColorBrewer::brewer.pal(9, "Blues")[c(3, 5, 7, 9)],
    label = Facet_Labels,
    guide = guide_legend(reverse = FALSE)
  ) +

  scale_y_continuous(name = "Percentage of each sample who reported given level of visit frequency",
                     labels = scales::percent) +

  xlab("Visit Frequency") +

  guides(fill = guide_legend(nrow = 2)) +

  theme(
    legend.position = "bottom",
    strip.background = element_rect(fill = "white"),
    strip.text =  TextSetup,
    legend.background = element_blank(),
    panel.grid.major.x = element_blank(),
    panel.grid.minor.x = element_blank(),
    panel.grid.major.y = element_blank(),
    panel.grid.minor.y = element_blank(),
    legend.text = TextSetup,
    legend.title = TextSetup,
    axis.text.x = element_blank(),
    axis.text.y = TextSetup,
    axis.title.y = TextSetup,
    axis.title.x = TextSetup
  )


# ***************************************************************************
#### Section 3: Export Plot ####
# ***************************************************************************


## Save with high DPI in the right place. I think png is the right format.
ggsave(
  FigureA1,
  device = "png",
  filename = paste0(
    here(),
    "/OtherOutput/Figures/",
    "H0_FigureA1_VisitFrequencyBySeasons.png"
  ),
  width = 25,
  height = 20,
  units = "cm",
  dpi = 500
)

# End Of Script -----------------------------------------------------------
