#### AllSeasons: Plot simulated mean WTP ####
# Script author: Peter King (p.king1@leeds.ac.uk)
# Last Edited: 14/10/2025
# Notes:
# - We don't simulate code here, just plot it.
# - The distribution is of the simulated means


# *****************************************************************************
#### Section 0: Setting up ####
## Find session information in 00_ReplicatePaper.R
# *****************************************************************************


## Libraries here: ************************************************************
library(apollo)
library(tidyverse)
library(dplyr)
library(magrittr)
library(mded)
library(here)
library(data.table)
library(Rfast)
library(tidyverse)


# ****************************************************************
#### Section 1: Import data ####
# ****************************************************************


## This is the very large data frame
Data <- here(
  "CEOutputData/H3",
  "SimulatedMeans_Correlated.csv"
) %>%
  fread() %>%
  data.frame()


# ****************************************************************
#### Section 2: Setup  ####
# ****************************************************************


### Define variable names
Names <- c("Colour2",
           "Colour",
           "Smell2",
           "Smell",
           "Sound2",
           "Sound",
           "Deadwood2",
           "Deadwood"
)


## Organise all simulated values
PlotData <- Data %>%
  dplyr::select(!Tax) %>%
  pivot_longer(1:8) %>%
  summarise(.by = c(name, Season),
            y0 = quantile(value, 0.025),
            y25 = quantile(value, 0.25),
            y50 = quantile(value, 0.50),
            y75 = quantile(value, 0.75),
            y100 = quantile(value, 0.975)) %>%
  mutate(name = factor(name, levels = Names))


# ****************************************************************
#### Section 2: Define useful variables ####
# ****************************************************************


## Label X axis of ggplot boxplots
Labels <- c(
  "Colours:\nhigh",
  "Colours:\nmedium",
  "Smells:\nhigh",
  "Smells:\nmedium",
  "Sounds:\nhigh",
  "Sounds:\nmedium",
  "Deadwood\ndecomposition:\nhigh",
  "Deadwood\ndecomposition:\nmedium"

)


## Label grouping variable
Seasons <- c("Winter", "Spring", "Summer", "Autumn")


## Define here so that we can change everything in one go
TextSize <- 12


TextSetup <- element_text(size = TextSize,
                          colour = "black",
                          family = "serif")


# ****************************************************************
#### Section 3: Create Plot ####
# ****************************************************************


## Using this version now!
Figure2 <- PlotData %>%

  ggplot(aes(x = name,
             fill = as.factor(Season))) +

  geom_errorbar(aes(
    ymin = y0,
    ymax = y100),
    width = 0.5,
    position = position_dodge(1)) +

  geom_boxplot(position = position_dodge(1),
               varwidth = 0.5,
               outlier.shape = NA,
               aes(
                 ymin = y0,
                 lower = y25,
                 middle = y50,
                 upper = y75,
                 ymax = y100,
               ),
               stat = "identity" ## means you can specify moments as in AES()
  ) +

  theme_bw() +

  ylab("Mean marginal WTP (GBP) in local tax,\nper household, per annum") +

  scale_y_continuous(breaks = seq(-15, 35, 5)) +

  scale_x_discrete(name = "Attribute and level",
                   labels = Labels) +

  scale_fill_manual(
    name = "Season",
    values = RColorBrewer::brewer.pal(9, "Blues")[c(2, 4, 6, 8)],

    label = c(
      "Winter\n (n=1879)\n",
      "Spring\n (n=1769)\n",
      "Summer\n (n=1767)\n",
      "Autumn\n (n=1748)\n"),

    guide = guide_legend(reverse = FALSE)) +

  geom_hline(yintercept = 0, linetype = "dashed", alpha = 0.25) +
  geom_vline(xintercept = 1.5, alpha = 0.25) +
  geom_vline(xintercept = 2.5, alpha = 0.25) +
  geom_vline(xintercept = 3.5, alpha = 0.25) +
  geom_vline(xintercept = 4.5, alpha = 0.25) +
  geom_vline(xintercept = 5.5, alpha = 0.25) +
  geom_vline(xintercept = 6.5, alpha = 0.25) +
  geom_vline(xintercept = 7.5, alpha = 0.25) +

  theme(
    legend.position = "bottom",
    legend.background = element_blank(),

    panel.grid.major.x = element_blank(),
    panel.grid.minor.x = element_blank(),
    panel.grid.major.y = element_blank(),
    panel.grid.minor.y = element_blank(),

    axis.text.x = TextSetup,
    axis.text.y = TextSetup,
    axis.title.y = TextSetup,
    axis.title.x = TextSetup,
    legend.text = TextSetup,
    legend.title = TextSetup

  )


# ****************************************************************
#### Section 3: Export Plot ####
# ****************************************************************


## Export and save in the right location
ggsave(
  Figure2,
  device = "png",
  filename = here("OtherOutput/Figures",
                  "H3_Figure2_DistributionSimulatedWTPBySeason.png"),
  width = 25,
  height = 15,
  units = "cm",
  dpi = 500
)


# End Of Script # ****************************************************************
