#### AllSeasons: Figure2_ScaleVariance ####
## Function: Creates Figure2_ScaleVariance but option A/B are merged
## Author: Dr Peter King (p.king1@leeds.ac.uk)
## Last change: 16/010/2025
## Changes:
# - Rewrote nicely
# - Updated labels
# - trim Y axis white space
# - adjust blues


# *****************************************************************************
#### Section 0: Setting up ####
## Find session information in 00_ReplicatePaper.R
# *****************************************************************************


## Libraries here: ************************************************************
library(magrittr)
library(dplyr)
library(apollo)
library(reshape2)
library(ggplot2)
library(ggridges)
library(distributional)
library(ggdist)
library(gridExtra)
library(data.table)
library(here)
library(tidyverse)
library(RColorBrewer)
library(scales)


# ****************************************************************
#### Section 1: Import Data ####
# ****************************************************************


## Output one row per choice format i.e., N=7163 * 9 tasks
Data <-
  here("CEInputData", "database_AllSeasons_Step1_Anonymised.csv") %>%
  fread() %>%
  data.frame()


## Reshape and isolate choices
AllSeasons <- Data %>%
  # ensure Task is integer 1..9
  mutate(Task = as.integer(as.character(Task))) %>%
  # sanity: keep respondent-level vars from the first row (should be identical)
  group_by(Respondent) %>%
  mutate(
    Season = first(Season),
    MostRecentVisit = first(MostRecentVisit)
  ) %>%
  ungroup() %>%
  # pivot: Task -> Choice1..Choice9
  pivot_wider(
    id_cols = c(Respondent, Season, MostRecentVisit),
    names_from = Task,
    values_from = Choice,
    names_prefix = "Choice"
  ) %>%
  # guarantee column order
  dplyr::select(Respondent, Season, MostRecentVisit, paste0("Choice", 1:9))



# ****************************************************************
#### Section 2: Reformat data to have values per choice per season  ####
## just laying %>% here
# ****************************************************************


# ---- compute season totals dynamically ----
season_totals <- AllSeasons %>%
  as_tibble() %>%
  distinct(Respondent, Season) %>%
  dplyr::count(Season, name = "n_resp") %>%
  mutate(Total = n_resp * 9) %>%
  dplyr::select(Season, Total, n_resp)




# dynamically create legend labels: "Winter\n(n=1879)" etc.
season_labels <- season_totals %>%
  arrange(Season) %>%
  mutate(label = paste0(
    c("Winter", "Spring", "Summer", "Autumn"),
    "\n(n=", n_resp, ")"
  )) %>%
  pull(label)



# ---- prepare summary table ----
## Here, we reshape the data and then count how many choices are for each alternative
## We then calculate percentage of choices by season
## And then calculate quantiles to plot the boxplots
PlotData <-
  AllSeasons %>%

  as_tibble() %>%

  pivot_longer(
    cols = starts_with("Choice"),
    names_to = "Task",
    values_to = "value"
  ) %>%

  mutate(value = as.integer(as.character(value))) %>%

  mutate(value = case_when(
    value %in% c(1L, 2L) ~ "Yes",    # OPTION A or B
    value == 3L            ~ "No",   # NO CHANGE
    TRUE ~ as.character(value)
  )) %>%

  group_by(Season, value) %>%

  summarise(Freq = n(), .groups = "drop") %>%

  left_join(season_totals, by = "Season") %>%

  mutate(
    YMIN_count = pmax(0, Freq - 1.96 * sqrt(Freq)),
    YMAX_count = Freq + 1.96 * sqrt(Freq),
    Y25_count  = pmax(0, Freq - (1.96 / 2) * sqrt(Freq)),
    Y75_count  = Freq + (1.96 / 2) * sqrt(Freq),

    Percentage = Freq / Total,
    YMIN = YMIN_count / Total,
    YMAX = YMAX_count / Total,
    Y25  = Y25_count  / Total,
    Y75  = Y75_count  / Total
  ) %>%

  dplyr::select(Season, value, Freq, Total, n_resp, Percentage, YMIN, Y25, Y75, YMAX)



# ****************************************************************
#### Section 3A: Plot Setup ####
# ****************************************************************

TextSize <- 12

TextSetup <- element_text(size = TextSize,
                          colour = "black",
                          family = "serif")


# ****************************************************************
#### Section 3B: Create Plot ####
# ****************************************************************


## Create the figure bit by bit
FigureB1 <-
  PlotData %>%

  ggplot(aes(x = factor(Season), y = Percentage, fill = factor(Season))) +

  geom_errorbar(aes(ymin = YMIN, ymax = YMAX),
                width = 0.25,
                position = position_dodge(width = 0.8)) +

  geom_boxplot(aes(ymin = YMIN, lower = Y25, middle = Percentage, upper = Y75, ymax = YMAX),
               varwidth = 0.5,
               width = 0.8,
               outlier.shape = NA,
               stat = "identity") +

  facet_wrap(~ value,
             labeller = as_labeller(c("No" = "No Change", "Yes" = "Option A or B")),
             scales = "free_y") +

  theme_bw() +

  scale_y_continuous(
    name = "Percentage of Choices",
    labels = label_percent(accuracy = 1),    # whole-number percentages, no decimals
    breaks = pretty_breaks(n = 6)            # sensible evenly spaced major ticks
  ) +

  scale_x_discrete(name = NULL, labels = NULL) +

  scale_fill_manual(
    name = "Season",
    values = brewer.pal(9, "Blues")[c(2, 4, 6, 8)],
    labels = season_labels,
    guide = guide_legend(reverse = FALSE)
  ) +

  theme(
    legend.position = "bottom",
    legend.text = TextSetup,
    legend.title = TextSetup,
    axis.text.x = TextSetup,
    axis.text.y = TextSetup,
    axis.title.y = TextSetup,
    strip.background = element_rect(fill = "white"),
    strip.text = TextSetup,
    legend.background = element_blank(),
    panel.grid.major.x = element_blank(),
    panel.grid.minor.x = element_blank(),
    panel.grid.major.y = element_blank()
  )


# ****************************************************************
#### Section 4: Export Plot ####
# ****************************************************************


ggsave(
  FigureB1,
  device = "png",
  filename = here("OtherOutput/Figures",
                  "H2_FigureB1_ChoiceVarianceBySeason.png"),
  width = 20,
  height = 15,
  units = "cm",
  dpi = 500
)


# End Of Script -----------------------------------------------------------
