#### AllSeasons: Table A3 Likelihood-Ratio tests with balanced models  ####
## Author: Dr Peter King (p.king1@leeds.ac.uk)
## Relates to:
# "Table 2 Log-likelihoods and p values for
# Likelihood-ratio tests of Hypothesis One (equal preference parameters)
# comparing mixed logits estimated for each season."
## Last change: 16/10/2025
# - Used both pooled models _ScaleFixed and _ScaleVariable
# - Fixed AddStars() formatting
# - New scale variable model name
# - Reformatted TableA3 middle
# - now using correlated models


# *****************************************************************************
#### Section 0: Setting up ####
## Find session information in 00_ReplicatePaper.R
# *****************************************************************************


## Libraries here: ************************************************************
rm(list = ls())
library(here)
# library(DCchoice)
library(lubridate)
library(tidyr)
library(apollo)
library(ggridges)
library(ggplot2)
library(reshape2)
library(dplyr)
library(magrittr)
library(ggdist)
library(RColorBrewer)
library(data.table)
library(lmtest)


# ******************************************************************************
# Section 1: Import Data ####
# Selected output of 'sessionInfo()'
# ******************************************************************************


# Model_Winter <- here("CEOutputData/H1/Correlated", "H1_Winter_PreferenceSpaceModel_Correlated_model.rds") %>% readRDS()
Model_Winter <- here("CEOutputData/H1/Balanced", "H1_Winter_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Model_Spring <- here("CEOutputData/H1/Balanced", "H1_Spring_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Model_Summer <- here("CEOutputData/H1/Balanced", "H1_Summer_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Model_Autumn <- here("CEOutputData/H1/Balanced", "H1_Autumn_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Model_AllSeasons_ScaleFixed <- here("CEOutputData/H1/Balanced", "H1_AllSeasons_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Model_AllSeasons_ScaleVariable <- here("CEOutputData/H2/Balanced", "H2_WinterFixed_ScaleModel_Balanced_Correlated_model.rds") %>% readRDS()


# ******************************************************************************
# Section 2: Describe LR test ####
## Problem: apollo::apollo_lrtest requires same n. obs for test/retest
## Problem: lmtest:lrtest doesn't like apollo objects
## Solution: Custom LR function as per: https://doi.org/10.1016/j.ecolecon.2017.03.031
## Relevant text here:

# We follow the LR procedure detailed by Swait and Louviere (1993)
# in which the data from test and retest is stacked and
# a pooled model (AllSeasons) is estimated.
# The likelihood ratio (LR) test statistic is calculated as follows:

# LR = -2 (LL_Pooled - (LL_Test + LL_Retest))
# P ~ X2(df = NParams)

# The LR statistic has a Chi -square distribution with
# degrees of freedom equal to the number of parameters in the utility function.
# The LR is an asymptotic test of global goodness of fit and
# it tells us whether the
# variables with restricted (to be equal across waves) coefficients explain
# the same amount of variance before or after the restriction.
# If LR statistic does not exceed the 5% critical value
# we can conclude that the models for test and retest are sufficiently similar.

# The less restrictive test LR involves including
# explicit SCALE PARAMETERS in the pooled model to allow for
# differences in relative scale across waves.
# ******************************************************************************


# ******************************************************************************
# LR Test for Variable Scale model


## So calculate Likelihood ratio here
LR_Variable <-
  -2 * (
    Model_AllSeasons_ScaleVariable$finalLL %>% as.numeric() -
      (Model_Winter$LLout %>% as.numeric() +
         Model_Spring$LLout %>% as.numeric() +
         Model_Summer$LLout %>% as.numeric() +
         Model_Autumn$LLout %>% as.numeric()
      )) %>%
  as.numeric()


## Degrees of freedom
DF_Variable <- Model_AllSeasons_ScaleVariable$numParams


## P values here
### but need to add stars
p_Variable <- stats::pchisq(LR_Variable,
                            DF_Variable,
                            lower.tail = FALSE)

## Construct left to right
TableA3_Left_Variable <- rbind("Winter",
                              "Spring",
                              "Summer",
                              "Autumn",
                              "AllSeasons_ScaleVariable")


# Construct model names
model_names_Variable <- paste0("Model_",
                               TableA3_Left_Variable)

# Create a list of models
models_Fixed <- lapply(model_names_Variable,
                       get)


# Extract values using vectorized operations
N_values_Variable <-
  lapply(models_Fixed, function(x)
    x$nIndivs) %>%
  unlist()


## Log-likelihood
LogLik_values_Variable <-
  lapply(models_Fixed, function(x)
    x$finalLL) %>% unlist()

## AIC
AIC_values_Variable <-
  lapply(models_Fixed, function(x)
    x$AIC) %>% unlist()


## Adjusted R2
AdjR_values_Variable <-
  lapply(models_Fixed, function(x)
    x$adjRho2_0) %>% unlist()


# Diagnostic information for all models
TableA3_Middle_Variable <- data.frame(
  N          = N_values_Variable,
  AIC        = AIC_values_Variable,
  AdjustedR2 = AdjR_values_Variable,
  LogLik     = LogLik_values_Variable
) %>%
  dplyr::mutate(dplyr::across(everything(), ~ sprintf("%.3f", round(.x, 3))))


## Likelihood-ratio test and P value
TableA3_Right_Variable <- paste0(LR_Variable %>%
                                  round(3) %>%
                                  sprintf("%.3f", .),
                                " (df = ",
                                DF_Variable,
                                ")",
                                " (P: ",
                                p_Variable %>% round(3) %>% sprintf("%.3f", .),
                                ")")


## Use rep() to force rows to match but delete when in Word
TableA3_Variable <- bind_cols(
  TableA3_Left_Variable,
  TableA3_Middle_Variable,
  TableA3_Right_Variable)



# ******************************************************************************
# LR Test for Fixed Scale model

## So calculate Likelihood ratio here
LR_Fixed <-
  -2 * (
    Model_AllSeasons_ScaleFixed$finalLL %>% as.numeric() -
      (Model_Winter$LLout %>% as.numeric() +
         Model_Spring$LLout %>% as.numeric() +
         Model_Summer$LLout %>% as.numeric() +
         Model_Autumn$LLout %>% as.numeric()
      )) %>%
  as.numeric()


## Degrees of freedom
DF_Fixed <- Model_AllSeasons_ScaleFixed$numParams


## P values here
### but need to add stars
p_Fixed <- stats::pchisq(LR_Fixed,
                         DF_Fixed,
                         lower.tail = FALSE)


## Construct left to right
TableA3_Left_Fixed <- rbind("Winter",
                           "Spring",
                           "Summer",
                           "Autumn",
                           "AllSeasons_ScaleFixed")



# Construct model names
model_names_Fixed <- paste0("Model_",
                            TableA3_Left_Fixed)

# Create a list of models
models_Fixed <- lapply(model_names_Fixed,
                       get)


# Extract values using vectorized operations
# Extract values using vectorized operations
N_values_Fixed <-
  lapply(models_Fixed, function(x)
    x$nIndivs) %>% unlist()


## Log-likelihood
LogLik_values_Fixed <-
  lapply(models_Fixed, function(x)
    x$finalLL) %>% unlist()

## AIC
AIC_values_Fixed <-
  lapply(models_Fixed, function(x)
    x$AIC) %>% unlist()


## Adjusted R2
AdjR_values_Fixed <-
  lapply(models_Fixed, function(x)
    x$adjRho2_0) %>% unlist()


# Create a data frame
TableA3_Middle_Fixed <- data.frame(N = N_values_Fixed,
                                  LogLik = LogLik_values_Fixed %>%
                                    round(3) %>%
                                    sprintf("%.3f", .))


# Diagnostic information for all models
TableA3_Middle_Fixed <- data.frame(
  N = N_values_Fixed,
  AIC = AIC_values_Fixed,
  AdjustedR2 = AdjR_values_Fixed,
  LogLik = LogLik_values_Fixed
) %>%
  dplyr::mutate(dplyr::across(everything(), ~ sprintf("%.3f", round(.x, 3))))


## Likelihood-ratio test and P value
TableA3_Right_Fixed <- paste0(LR_Fixed %>%
                               round(3) %>%
                               sprintf("%.3f", .),
                             " (df = ",
                             DF_Fixed,
                             ")",
                             " (P: ",
                             p_Fixed %>% round(3) %>% sprintf("%.3f", .),
                             ")")


## Use rep() to force rows to match but delete when in Word
TableA3_Fixed <- bind_cols(
  TableA3_Left_Fixed,
  TableA3_Middle_Fixed,
  TableA3_Right_Fixed)


# ******************************************************************************
# Section 5: Export ####
# ******************************************************************************


## I'm stitching JUST the last row of the variable model to avoid repetition
TableA3 <- rbind(TableA3_Fixed,
                TableA3_Variable[5, ])



## Set correct row names
TableA3$...1 <-  c(
  "Winter",
  "Spring",
  "Summer",
  "Autumn",
  "All seasons pooled: fixed scale",
  "All seasons pooled: variable scale"
)


## Set correct column names
colnames(TableA3) <- c(
  "Season",
  "Participants",
  "AIC",
  "Adjusted R2",
  "Log-likelihood",
  "Test statistic"
)


## TableA3 to screen ----------------------------------------
TableA3 %>%
  write.csv(quote = FALSE,
            row.names = FALSE)


## TableA3 to file ----------------------------------------
TableA3 %>% fwrite(sep = ",",
                  quote = FALSE,
                  file = paste0(here(
                    "OtherOutput/Tables",
                    "TableA3_H1_LRtests_Balanced.txt"
                  )))



# End of script **********************************************************
