#### AllSeasons: Table A5 modeloutputs for prefspace balanced models  ####
## Author: Dr Peter King (p.king1@leeds.ac.uk)
## Relates to:
# "Table 11 Weighted Mixed Logit Models."
## Last change: 16/10/2025
# - Added _ScaleVariable models
# - Rearranged modeloutput


# *****************************************************************************
#### Section 0: Setting up ####
## Find session information in 00_ReplicatePaper.R
# *****************************************************************************


## Libraries here: ************************************************************
options(scipen = 90)
library(apollo)
library(dplyr)
library(magrittr)
library(mded)
library(here)
library(data.table)


# ******************************************************************************
#### Section 1: Import Data ####
# ******************************************************************************


# ******************************************************************************
# Section 1A: Import Data ####
# Selected output of 'sessionInfo()'
# ******************************************************************************


Winter_Model <- here("CEOutputData/H1/Balanced", "H1_Winter_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Spring_Model <- here("CEOutputData/H1/Balanced", "H1_Spring_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Summer_Model <- here("CEOutputData/H1/Balanced", "H1_Summer_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
Autumn_Model <- here("CEOutputData/H1/Balanced", "H1_Autumn_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
AllSeasons_Model <- here("CEOutputData/H1/Balanced", "H1_AllSeasons_PreferenceSpaceModel_Correlated_Balanced_model.rds") %>% readRDS()
AllSeasons_Model_ScaleVariable <- here("CEOutputData/H2/Balanced", "H2_WinterFixed_ScaleModel_Balanced_Correlated_model.rds") %>% readRDS()


# ******************************************************************************
# Section 1B : Import Data ####
# Selected output of 'sessionInfo()'
# ******************************************************************************


Winter_Estimates <- here("CEOutputData/H1/Balanced", "H1_Winter_PreferenceSpaceModel_Correlated_Balanced_estimates.csv") %>% fread() %>% data.frame()
Spring_Estimates <- here("CEOutputData/H1/Balanced", "H1_Spring_PreferenceSpaceModel_Correlated_Balanced_estimates.csv") %>% fread() %>% data.frame()
Summer_Estimates <- here("CEOutputData/H1/Balanced", "H1_Summer_PreferenceSpaceModel_Correlated_Balanced_estimates.csv") %>% fread() %>% data.frame()
Autumn_Estimates <- here("CEOutputData/H1/Balanced", "H1_Autumn_PreferenceSpaceModel_Correlated_Balanced_estimates.csv") %>% fread() %>% data.frame()
AllSeasons_Estimates <- here("CEOutputData/H1/Balanced", "H1_AllSeasons_PreferenceSpaceModel_Correlated_Balanced_estimates.csv") %>% fread() %>% data.frame()
AllSeasons_Estimates_ScaleVariable <- here("CEOutputData/H2/Balanced", "H2_WinterFixed_ScaleModel_Balanced_Correlated_estimates.csv") %>% fread  %>% data.frame()


# Rename the inconsistent covariate names
# AllSeasons_Estimates_ScaleVariable$V1 <- gsub("beta_Impairment_Sight", "beta_Impairment_Colour",
#                                               AllSeasons_Estimates_ScaleVariable$V1)
# AllSeasons_Estimates_ScaleVariable$V1 <- gsub("beta_Impairment_Hearing", "beta_Impairment_Sound",
#                                               AllSeasons_Estimates_ScaleVariable$V1)


# ******************************************************************************
#### Section 2: Define table-making functions ####
# ******************************************************************************


# Define variable ordering
BaseVariables <- c(
  "asc_C",

  # Means
  "mu_Tax", "mu_Colour2", "mu_Colour", "mu_Smell2", "mu_Smell",
  "mu_Sound2", "mu_Sound", "mu_Deadwood2", "mu_Deadwood",

  # Standard deviations
  "sig_Tax", "sig_Colour2", "sig_Colour", "sig_Smell2", "sig_Smell",
  "sig_Sound2", "sig_Sound", "sig_Deadwood2", "sig_Deadwood",

  # Correlations (you'll need to specify the exact order you want)
  grep("^sig_.*_", Winter_Estimates$V1, value = TRUE),

  # Covariates
  "beta_Age",
  "beta_CountryDummy",
  "beta_MilesDistance",
  "beta_EthnicityDummyWhite",
  "beta_Gender",
  "beta_Income",
  "beta_Impairment_Colour",
  "beta_Impairment_Smell",
  "beta_Impairment_Sound",
  "beta_Urbanicity",
  "beta_MostRecentVisit"
)


# Remove duplicates from correlation grep
BaseVariables <- unique(BaseVariables)


# Season-specific parameters (only in pooled model)
SeasonParameters <- c("mu_Winter", "mu_Spring", "mu_Summer", "mu_Autumn")


# Full ordering
OutputOrder_Season <- c(BaseVariables, SeasonParameters)


# Modified function that handles missing variables
ModelOutput <- function(Estimates) {
  stars <- cut(Estimates$Rob.p.val.0.,
               breaks = c(-Inf, 0.01, 0.05, 0.1, Inf),
               labels = c("***", "**", "*", ""),
               right = FALSE)

  result <- data.frame(
    Variable = Estimates$V1,
    Estimate = paste0(
      sprintf("%.3f", Estimates$Estimate), stars,
      " (", sprintf("%.3f", Estimates$Rob.std.err.), ")"
    )
  )

  # Match to OutputOrder_Season, keeping NA for missing variables
  matched <- result[match(OutputOrder_Season, result$Variable), ]
  matched$Variable <- OutputOrder_Season
  matched$Estimate[is.na(matched$Estimate)] <- ""

  return(matched)
}


## Summarise diagnostics per model
Diagnostics <- function(Model) {
  rbind(
    "N" = Model$nIndivs,
    "AIC" = sprintf("%.3f", Model$AIC),
    "Adj.R2" = sprintf("%.3f", Model$adjRho2_0),
    "Log-Likelihood" = sprintf("%.3f", Model$finalLL)
  )
}


## Tidying variable names
CleanVariableNames <- function(var_names) {
  sapply(var_names, function(x) {
    # ASC
    if (x == "asc_C") return("ASC")

    # Means
    if (x == "mu_Tax") return("Tax")
    if (x == "mu_Colour2") return("Colours: high")
    if (x == "mu_Colour") return("Colours: medium")
    if (x == "mu_Smell2") return("Smells: high")
    if (x == "mu_Smell") return("Smells: medium")
    if (x == "mu_Sound2") return("Sounds: high")
    if (x == "mu_Sound") return("Sounds: medium")
    if (x == "mu_Deadwood2") return("Deadwood decomposition: high")
    if (x == "mu_Deadwood") return("Deadwood decomposition: medium")

    # Standard deviations
    if (x == "sig_Tax") return("Tax")
    if (x == "sig_Colour2") return("Colours: high")
    if (x == "sig_Colour") return("Colours: medium")
    if (x == "sig_Smell2") return("Smells: high")
    if (x == "sig_Smell") return("Smells: medium")
    if (x == "sig_Sound2") return("Sounds: high")
    if (x == "sig_Sound") return("Sounds: medium")
    if (x == "sig_Deadwood2") return("Deadwood decomposition: high")
    if (x == "sig_Deadwood") return("Deadwood decomposition: medium")

    # Correlations
    if (grepl("^sig_.*_", x)) {
      # Remove "sig_" prefix
      clean <- sub("^sig_", "", x)
      # Split on underscore
      parts <- strsplit(clean, "_")[[1]]

      # Helper to format attribute names
      format_attr <- function(attr) {
        # Handle numbered versions
        attr <- gsub("1$", "", attr)  # Remove trailing "1"

        if (grepl("2$", attr)) {
          attr <- sub("2$", "", attr)
          level <- ": high"
        } else {
          level <- ": medium"
        }

        # Map to readable names
        base <- switch(attr,
                       "Sound" = "Sounds",
                       "Smell" = "Smells",
                       "Colour" = "Colours",
                       "Deadwood" = "Deadwood decomposition",
                       attr)

        return(paste0(base, level))
      }

      return(paste(format_attr(parts[1]), "*", format_attr(parts[2])))
    }

    # Covariates (remove beta_ prefix)
    if (grepl("^beta_", x)) {
      clean <- sub("^beta_", "", x)
      return(switch(clean,
                    "Age" = "Age (younger than sample median age)",
                    "Charity" = "Charity membership (involved in any relevant charity)",
                    "Country" = "Country of residence (England)",
                    "Distance" = "Distance to nearby forest (distance levels in kilometres)",
                    "White" = "Ethnicity (White)",
                    "Gender" = "Gender (man)",
                    "Income" = "Household income (greater than or equal to sample median income)",
                    "WoodlandsScore" = "Perceived biodiversity score",
                    "Impairment_Colour" = "Sensory visual impairment",
                    "Impairment_Smell" = "Sensory smell impairment",
                    "Impairment_Sound" = "Sensory hearing impairment",
                    "Urbanicity" = "Urbanicity (urban)",
                    "MostRecentVisit" = "Visit frequency (reported visits to forests per season)",
                    clean))
    }

    # Scale variables
    if (x == "mu_Winter") return("Scale: Winter")
    if (x == "mu_Spring") return("Scale: Spring")
    if (x == "mu_Summer") return("Scale: Summer")
    if (x == "mu_Autumn") return("Scale: Autumn")

    # Diagnostics
    if (x == "N") return("N")
    if (x == "AIC") return("AIC")
    if (x == "Adj.R2") return("Adj.R²")
    if (x == "Log-Likelihood") return("Log-Likelihood")

    # Default: return as-is
    return(x)
  }, USE.NAMES = FALSE)
}




# ******************************************************************************
#### Section 3: Make table ####
# ******************************************************************************


# Adds the model outputs
TableA5_Top <- data.frame(
  Variable = ModelOutput(Winter_Estimates)[, 1] %>% gsub(pattern = "beta_", replacement = ""),
  Winter = ModelOutput(Winter_Estimates)[, 2],
  Spring = ModelOutput(Spring_Estimates)[, 2],
  Summer = ModelOutput(Summer_Estimates)[, 2],
  Autumn = ModelOutput(Autumn_Estimates)[, 2],
  `Pooled: scale fixed` = ModelOutput(AllSeasons_Estimates)[, 2],
  `Pooled: scale variable` = ModelOutput(AllSeasons_Estimates_ScaleVariable)[, 2],
  check.names = FALSE,
  stringsAsFactors = FALSE
)

# Adds the diagnostics to the bottom part of the table
TableA5_Bottom <- data.frame(
  Variable = c("N", "AIC", "Adj.R2", "Log-Likelihood"),
  Winter = Diagnostics(Winter_Model),
  Spring = Diagnostics(Spring_Model),
  Summer = Diagnostics(Summer_Model),
  Autumn = Diagnostics(Autumn_Model),
  `Pooled: scale fixed` = Diagnostics(AllSeasons_Model),
  `Pooled: scale variable` = Diagnostics(AllSeasons_Model_ScaleVariable),
  check.names = FALSE,
  stringsAsFactors = FALSE
)

## Add together
TableA5_Complete <- rbind(TableA5_Top, TableA5_Bottom)

# Apply clean names
TableA5_Complete$Variable <- CleanVariableNames(TableA5_Complete$Variable)


# ******************************************************************************
#### Section 4: Export results ####
# ******************************************************************************


## TableA5 to screen ----------------------------------------
TableA5_Complete %>%
  write.csv(quote = FALSE,
            row.names = FALSE)


## TableA5 to file ----------------------------------------
TableA5_Complete %>% data.frame() %>%
  fwrite(sep = ",",
         quote = FALSE,
         file = paste0(here(
           "OtherOutput/Tables",
           "TableA5_H1_ModelOutputs_Balanced.txt"
         )))

# End Of Script ------------------------------------------------
