#### AllSeasons: Table2  ###############
# Script author: Peter King (p.king1@leeds.ac.uk)
# Relates to:
# "Table 2 Wald test statistics and
# p values for Hypothesis Two (test of equal scale parameters).
# Results from a pooled mixed logit where the scale parameter
# for responses in winter was fixed at one,
# while the scale parameter for other seasons was allowed to vary. "
# Last Edited: 14/10/2025
# - Rewrote for correct indexing


# *****************************************************************************
#### Section 0: Setting up ####
## Find session information in 00_ReplicatePaper.R
# *****************************************************************************


## Libraries here: ************************************************************
options(scipen = 90)
library(dplyr)
library(magrittr)
library(mded)
library(here)
library(aod)
library(lmtest)
library(tidyverse)
library(data.table)


# ******************************************************************************
#### Section 1: Import Data ####
### WATCH OUT: FILE SIZE EXTREMELY LARGE (>2GB)
## Only need two models here, winter and spring fixed
# ******************************************************************************


## Most important model, the Winter scale =1, others vary
WinterFixed_Model <- here("CEOutputData/H2/Correlated",
                          "H2_WinterFixed_ScaleModel_FullSample_Correlated_Covariates_model.rds") %>%
  readRDS()


# 1) Build a safe tibble of estimates (term, value)
WinterFixed_Model_Estimates <- as.data.frame(WinterFixed_Model$estimate) %>%
  tibble::rownames_to_column(var = "term") %>%
  dplyr::rename(estimate = 2)          # second column holds numeric estimate


# ******************************************************************************
#### Section 2: Collate all variables in one place ####
## Aim: Extract relevant data from the model objects
# ******************************************************************************



# 2) Remove nuisance rows (explicit, robust)
CoefsToDrop <- c("asc_A", "asc_B", "mu_Winter")   # change if needed
WinterFixed_Model_Estimates <- WinterFixed_Model_Estimates %>% dplyr::filter(!term %in% CoefsToDrop)



# 3) Convert to named numeric vector b (preserves order in WinterFixed_Model_Estimates)
b <- set_names(WinterFixed_Model_Estimates$estimate, WinterFixed_Model_Estimates$term)


# 4) Subset and reorder V to match b (very important)
V_full <- WinterFixed_Model$varcov
common <- intersect(names(b), rownames(V_full))
if (length(common) != length(b)) {
  stop("Mismatch between estimate names and varcov rownames. Inspect names(b) and rownames(V).")
}


V <- V_full[common, common, drop = FALSE]
b <- b[common]   # ensure same order


# 5) compute indices for the mu parameters we need
Scale_Names <- c("mu_Spring", "mu_Summer", "mu_Autumn")
Scale_Index <- map_int(Scale_Names, ~ which(names(b) == .x))
Scale_Values <- map_dbl(Scale_Names, ~ as.numeric(WinterFixed_Model$estimate[.x]))   # original estimate extraction


# ******************************************************************************
#### Section 3: Construct Table ####
# I apologise for nothing, here's the table in the ugliest way possible
# ******************************************************************************


# 6) helper to run wald test by parameter position (single-term H0 = 1)
Scale_WaldTest <- function(pos) {
  res <-
    aod::wald.test(
      Sigma = V,
      b = b,
      Terms = pos,
      H0 = 1
    )$result$chi2
  tibble(stat = as.numeric(res[1]), pval = as.numeric(res[3]))
}


# 7) Build tidy table
Table2 <- tibble(
  `Test sample`    = "Winter",
  `Retest samples` = c("Spring", "Summer", "Autumn"),
  mu_name          = Scale_Names,
  mu_val           = Scale_Values
) %>%
  mutate(
    w = map(Scale_Index, Scale_WaldTest)
  ) %>%
  unnest(w) %>%
  mutate(
    `Scale parameter` = round(mu_val, 3),
    stat_round = sprintf("%.3f", round(stat, 3)),
    stars = case_when(
      pval < 0.01  ~ "***",
      pval < 0.05  ~ "**",
      pval < 0.10  ~ "*",
      TRUE         ~ ""
    ),
    `Wald test statistic` = paste0(stat_round, stars)
  ) %>%
  dplyr::select(`Test sample`, `Retest samples`, `Scale parameter`, `Wald test statistic`) %>%
  mutate(`Scale parameter` = sprintf("%.3f", Scale_Values))



# ******************************************************************************
#### Section 4: Export table ####
## Here we just output to console and text
# ******************************************************************************


## Table2 to screen ----------------------------------------
Table2 %>% write.csv(quote = FALSE,
                     row.names = FALSE)


## Table2 to file ----------------------------------------
Table2 %>% data.table::fwrite(sep = ",",
                  quote = FALSE,
                  file = paste0(here(
                    "OtherOutput/Tables",
                    "Table2_H2_Waldtests.txt"
                  )))



# End of script **********************************************************

