#### AllSeasons: Simulate mean WTP  ###############
# AUTHOR: Peter King (p.king1@leeds.ac.uk)
# LAST CHANGE: 12/10/2025
# FUNCTION: To simulate means
# THANKS: Thanks RCSD!


# *****************************************************************************
#### Section 0: Setting up ####
## Find session information in 00_ReplicatePaper.R
# *****************************************************************************


## Libraries here: ************************************************************
options(scipen = 90)
library(tidyverse)
library(dplyr)
library(magrittr)
library(mded)
library(here)
library(data.table)
library(stats)

library(mvtnorm)
library(apollo)
library(cmdlr)

# *************************************************************
#### Section 1: Setup loop  ####
# *************************************************************

# Model <- here("CEOutputData/H3",
#               "H3_AllSeasons_WTPSpaceModel_model.rds") %>%
#   readRDS()

Simulator <- function(Model, Season) {

  # *************************************************************
  #### Section 2: Initialisation of loop inputs  ####
  # *************************************************************


  #Set up number of draws for the variance covariance matrix (10K is good)
  N_Reps_KrinskyRobb <- 10000

  #Set up number of draws for simulating WTP (1 million is good but long, so 100K might be ok)
  N_Reps_Draws = 100000

  names <-
    c("Tax",
      "Colour",
      "Smell",
      "Sound",
      "Deadwood",
      "Colour2",
      "Smell2",
      "Sound2",
      "Deadwood2")

  wtp_matrices <- lapply(names, function(x) matrix(0, nrow = N_Reps_KrinskyRobb, ncol = 1))
  draw_matrices <- lapply(names, function(x) rnorm(N_Reps_Draws, 0, 1))

  names(wtp_matrices) <- names
  names(draw_matrices) <- names


  # *************************************************************
  #### Section 3: Values from models  ####
  # *************************************************************


  # How many parameters
  NParams <- Model$estimate %>% length()

  ## Take estimates and cut off ASC_A/B which were fixed at zero and
  ## don't feature in the Variance-Covariance matrix
  mean_v = Model$estimate[3:NParams]


  #Load variance-covariance matrix for the same model
  covar = as.matrix(Model$robvarcov)


  ## Take draws from distribution
  model_draws = mvtnorm::rmvnorm(n = N_Reps_KrinskyRobb,
                         mean = mean_v,
                         sigma = covar)
  model_draws = as.matrix(model_draws)


  # *************************************************************
  #### Section 3: Simulation loop ####
  # *************************************************************



  wtp_results <- vector("list", N_Reps_KrinskyRobb)

  # Loop through each Krinsky-Robb iteration
  for (i in 1:N_Reps_KrinskyRobb) {
    # For each iteration, pre-allocate a list to store WTP for parameters
    wtp_results[[i]] <- vector("list", length(names))
    names(wtp_results[[i]]) <- names


    # Loop through each parameter
    for (param in names) {
      # Construct the names for the mean and standard deviation parameters
      mu_name <- paste0("mu_", param)
      sig_name <- paste0("sig_", param)
      draws_name <- param  # No need to prepend "Draws_" here

      # Calculate WTP for each parameter
      wtp_results[[i]][[param]] <- (model_draws[i, mu_name] + model_draws[i, sig_name] * draw_matrices[[draws_name]]) %>% mean()
    }

    # Loop through each parameter
    # for (param in names) {
    #   # Construct the names for the mean and standard deviation parameters
    #   mu_name <- paste0("mu_", param)
    #   sig_name <- paste0("sig_", param)
    #   draws_name <- paste0("Draws_", param)
    #
    #   # Calculate WTP for each parameter
    #   wtp_results[[i]][[param]] <- (model_draws[i, mu_name] + model_draws[i, sig_name] * get(draws_name)) %>% mean()
    # }
  }

  # Combine results for each iteration into a single data frame
  Output <- rbindlist(wtp_results, use.names = TRUE, fill = TRUE) %>% data.frame()
  Output$Season <- rep(Season, N_Reps_KrinskyRobb)
  return(Output)
}




# *************************************************************
#### Section 2: Run loop  ####
# *************************************************************



Loop_Winter <- here("CEOutputData/H3/Correlated", "H3_Winter_WTPSpaceModel_Correlated_model.rds") %>%
  readRDS() %>% Simulator(Season = 0)
Loop_Spring <- here("CEOutputData/H3/Correlated", "H3_Spring_WTPSpaceModel_Correlated_model.rds") %>%
  readRDS() %>% Simulator(Season = 1)
Loop_Summer <- here("CEOutputData/H3/Correlated", "H3_Summer_WTPSpaceModel_Correlated_model.rds") %>%
  readRDS() %>% Simulator(Season = 2)
Loop_Autumn <- here("CEOutputData/H3/Correlated", "H3_Autumn_WTPSpaceModel_Correlated_model.rds") %>%
  readRDS() %>% Simulator(Season = 3)
Loop_AllSeasons <- here("CEOutputData/H3/Correlated", "H3_AllSeasons_WTPSpaceModel_Correlated_model.rds") %>%
  readRDS() %>% Simulator(Season = 4)


# *************************************************************
#### Section 3: Combine and export outputs  ####
# *************************************************************


Combined_Means <-
  bind_rows(Loop_Winter,
            Loop_Spring,
            Loop_Summer,
            Loop_Autumn)


Combined_Means %>% fwrite(sep = ",",
                          quote = FALSE,
                          paste0(here(
                            "CEOutputData/H3",
                            "SimulatedMeans_Correlated.csv"
                          )))


## Pooled WTP
Loop_AllSeasons %>% fwrite(sep = ",",
                           quote = FALSE,
                           paste0(here(
                             "CEOutputData/H3",
                             "AllSeasons_Correlated.csv"
                           )))

# *************************************************************
#### Section 4: Distributions  ####
# *************************************************************

# Test <- here("CEOutputData/H1/Balanced",
#              "H1_Autumn_PreferenceSpaceModel_Balanced_model.rds") %>% readRDS() %>% Simulator(Season = 3)
# (-Test / -exp(Test$Tax)) %>%
#   reshape2::melt() %>%
#   dplyr::filter(!variable %in% c("Tax", "Season")) %>%
#   ggplot(aes(
#     y = variable,
#     x = value,
#     group = variable,
#     fill = variable
#   ))+
#   ggdist::stat_histinterval(normalize = "groups") +
#   geom_vline(xintercept = 0)+
#   theme_bw()
